/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "SKEAZ1284.h"

#include "uart.h"
#include "SPI.h"
#include "pin_config.h"

#define UART_PORT	UART2
#define SYSCLK		(DEFAULT_SYSTEM_CLOCK / 2)
#define UART_BAUD	19200
#define UART_PINS	PTD6_PTD7

#define DELAY		3000

#define SPI_PORT	SPI1
#define SPI_PINS	PTG4_5_6_7

/* Little Endian to Big Endian macro */
#define LE2BE_32(c)	(((c & 0xFF) << 24) | ((c & 0xFF00) << 8) | ((c & 0xFF0000) >> 8) | ((c & 0xFF000000) >> 24))

/* Error codes */
#define ERR_OK 	0x41
#define ERR_CRC	0x45

/* SPI communication definitions */
#define START_SYNC		0x70
#define READ_RESP		0x52
#define MST_RQT			0x55

typedef union
{
	uint8_t Byte[37]; /* Byte level access to the Phrase */
	struct
	{
		char PhraseType;											/* Type of received record (e.g. S0, S1, S5, S9...) */
		uint8_t PhraseSize; 										/* Phrase size (address + data + checksum) */
		uint8_t PhraseAddress[4]__attribute__ ((aligned (32))); 	/* Address, depending on the type of record it might vary */
		uint8_t PhraseData[32]__attribute__ ((aligned (32))); 									/* Maximum 32 data bytes */
		uint8_t PhraseCRC;											/* Checksum of size + address + data */
	}F;
}BootPhraseStruct;

void UART_get_phrase(BootPhraseStruct *BP);
void SPI_send_phrase(BootPhraseStruct *BP);

int main(void)
{
	BootPhraseStruct BP;
	uint8_t ERR = 0;

    /* Enable UART */
	/* Initialize pin settings for UART communication */
	UART_pin_settings(UART_PINS);
	/*Initialize UART communication */
	UART_Init(UART_PORT, SYSCLK, UART_BAUD);

	/* Enable SPI */
	SPI_pin_settings(SPI_PINS);
	SPI_init(SPI_PORT, MASTER);

    /* This for loop should be replaced. By default this loop allows a single stepping. */
    while(1){
    	/* Get UART phrase */
    	UART_get_phrase(&BP);

    	/* Synch before sending phrase */
    	do{
    		/* Send request */
    		SPI_transfer(SPI_PORT, START_SYNC);

    		/* Delay in-between frames to allow sync */
    		for(uint32_t i = 0; i < DELAY; ++i){};

    		/* Read answer */
    		ERR = SPI_transfer(SPI_PORT, READ_RESP);
    	}while(ERR != START_SYNC);

    	/* Send phrase through SPI */
    	SPI_send_phrase(&BP);

    	/* Poll SPI for ERR */
    	do{
    		/* Send request */
    		ERR = SPI_transfer(SPI_PORT, MST_RQT);

    		/* Delay in-between frames to allow sync */
    		for(uint32_t i = 0; i < DELAY; ++i){};

    		/* Read answer */
    		ERR = SPI_transfer(SPI_PORT, READ_RESP);
    	}while(!((ERR == ERR_OK) | (ERR == ERR_CRC)));

    	/* Send ERR through UART */
    	UART_PutChar(UART_PORT, ERR);
    }

    /* Never leave main */
    return 0;
}

void UART_get_phrase(BootPhraseStruct *BP){
	char type = 0;
	uint8_t length = 0;

	/* Get SREC line start ('S') */
	do{
	type = UART_GetChar(UART_PORT);
	if(type != 'S'){
		/* Signal a CRC error to start a new transmission */
		UART_PutChar(UART_PORT, ERR_CRC);
	}
	}while(type != 'S');

	/* Store phrase type */
	BP->F.PhraseType = UART_GetChar(UART_PORT);

	/* Get byte count */
	length = UART_GetChar(UART_PORT);
	BP->F.PhraseSize = length;

	/* Check records with a 24-bit address field */
	if((BP->F.PhraseType == '2') | (BP->F.PhraseType == '6') | (BP->F.PhraseType == '8')){
		/* Get address */
		for(uint8_t i = 0; i < 3; i++){
			BP->F.PhraseAddress[i] = UART_GetChar(UART_PORT);
		}
		length = length - 3;
	}else if((BP->F.PhraseType == '3') | (BP->F.PhraseType == '7') ){		/* Check records with a 32-bit address field */
		/* Get address */
		for(uint8_t i = 0; i < 4; i++){
			BP->F.PhraseAddress[i] = UART_GetChar(UART_PORT);
		}
		length = length - 4;
	}else{																	/* All the other records have a 16-bit address field */
		/* Get address */
		for(uint8_t i = 0; i < 2; i++){
			BP->F.PhraseAddress[i] = UART_GetChar(UART_PORT);
		}
		length = length - 2;
	}

	/* Get data stream */
	for(uint8_t i = 0; i < (length - 1); i++){
		BP->F.PhraseData[i] = UART_GetChar(UART_PORT);
	}

	/* Get CRC */
	BP->F.PhraseCRC = UART_GetChar(UART_PORT);
}

void SPI_send_phrase(BootPhraseStruct *BP){
	uint8_t length = 0;

	/* Send SREC line start ('S') */
	SPI_transfer(SPI_PORT, 'S');

	/* Send phrase type */
	SPI_transfer(SPI_PORT, BP->F.PhraseType);

	/* Send byte count */
	length = BP->F.PhraseSize;
	SPI_transfer(SPI_PORT, BP->F.PhraseSize);

	/* Check records with a 24-bit address field */
	if((BP->F.PhraseType == '2') | (BP->F.PhraseType == '6') | (BP->F.PhraseType == '8')){
		/* Send address */
		for(uint8_t i = 0; i < 3; i++){
			SPI_transfer(SPI_PORT, BP->F.PhraseAddress[i]);
		}
		length = length - 3;
	}else if((BP->F.PhraseType == '3') | (BP->F.PhraseType == '7') ){		/* Check records with a 32-bit address field */
		/* Send address */
		for(uint8_t i = 0; i < 4; i++){
			SPI_transfer(SPI_PORT, BP->F.PhraseAddress[i]);
		}
		length = length - 4;
	}else{																	/* All the other records have a 16-bit address field */
		/* Send address */
		for(uint8_t i = 0; i < 2; i++){
			SPI_transfer(SPI_PORT, BP->F.PhraseAddress[i]);
		}
		length = length - 2;
	}

	/* Send data stream */
	for(uint8_t i = 0; i < (length - 1); i++){
		SPI_transfer(SPI_PORT, BP->F.PhraseData[i]);
	}

	/* Send CRC */
	SPI_transfer(SPI_PORT, BP->F.PhraseCRC);
}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
